using System;
using System.Collections;
using Modules.DataRetrieval;
using UnityEngine;
using UnityEngine.Networking;

interface ISSEClient
{
    void Start();
    void Stop();
}

public class SSEClient<T> : ISSEClient where T : class
{
    private string _url;
    private MonoBehaviour _runner;
    private Coroutine _sseRoutine;
    private UnityWebRequest _request;
    private SseDownloadHandler _handler;

    /// <summary>
    /// Constructor for the SSEClient class.
    /// Initializes the URL and the MonoBehaviour runner for coroutine management.
    /// </summary>
    public SSEClient(string url, MonoBehaviour runner)
    {
        _url = url;
        _runner = runner;
    }


    /// <summary>
    /// Starts the SSE (Server-Sent Events) connection by initiating the coroutine responsible for handling the connection.
    /// If the SSE routine is not already running, it logs the start and begins the connection process.
    /// </summary>
    public void Start()
    {
        if (_sseRoutine == null)
            Debug.Log("Started SSE");
        _sseRoutine = _runner.StartCoroutine(Connect());
    }

    /// <summary>
    /// Stops the SSE connection by aborting the UnityWebRequest if it's still in progress,
    /// stopping the coroutine responsible for handling the connection,
    /// and unsubscribing from the message received event.
    /// Resets the routine and logs the stop action.
    /// </summary>
    public void Stop()
    {
        if (_request != null && !_request.isDone)
            _request.Abort();

        if (_sseRoutine != null)
            _runner.StopCoroutine(_sseRoutine);

        if (_handler != null)
            _handler.OnMessageReceived -= HandleMessage;

        _sseRoutine = null;
        Debug.Log("Stopped SSE");
    }

    /// <summary>
    /// Initiates the connection to the SSE server using UnityWebRequest.
    /// Sets up the request with the specified URL and download handler.
    /// Handles the response and errors during the connection process.
    /// The connection is kept alive until the request is done or an error occurs.
    /// </summary>
    /// <returns>An IEnumerator for coroutine handling.</returns>
    private IEnumerator Connect()
    {
        _request = UnityWebRequest.Get(_url);
        _handler = new SseDownloadHandler();
        _handler.OnMessageReceived += HandleMessage;
        _request.downloadHandler = _handler;
        _request.SetRequestHeader("Accept", "text/event-stream");
        _request.timeout = 0;

        var async = _request.SendWebRequest();

        while (!_request.isDone)
        {
            if (_request.result == UnityWebRequest.Result.ConnectionError ||
                _request.result == UnityWebRequest.Result.ProtocolError)
            {
                Debug.LogError("SSE Error: " + _request.error);
                break;
            }
            yield return null;
        }
    }

    /// <summary>
    /// Handles the message received from the SSE server.
    /// Parses the message using JsonUtility and raises an event with the parsed data.
    /// If parsing fails, logs the error message.
    /// </summary>
    /// <param name="msg">The message received from the SSE server.</param>
    /// <exception cref="Exception">Thrown when parsing the message fails.</exception>
    private void HandleMessage(string msg)
    {
        try
        {
            T data = JsonUtility.FromJson<T>(msg);
            SSEEventBus.RaiseDataReceived(_url, data);
        }
        catch (Exception e)
        {
            Debug.LogError("Parse Error: " + e.Message);
        }
    }
}